/* Program an EEPROM using my programmer
 *
 * Parallel Port Bit   D3          D2        D1                        D0
 * Function            Bit clock   Bit Data  Write enable (active low) +5V
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/io.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#define IODELAY 5e+3  /* System "clock" period.  Very dependent on system!!! 
		       * Should end up with iodelay() being about 1 ms */

void bitout( unsigned char bit );
void wordout( unsigned short address, unsigned char data );
void iodelay( void );
int killws( char *str );
unsigned str2num( char *str, char hexflag );

main( int argc, char *argv[] )
{
  FILE *fp;
  char hexflag, line[80];
  unsigned short addr;
  unsigned char data;

  addr = 0;

  if( argc < 2 )
    {
      fprintf( stderr, "usage: eeprom [-h] file\n" );
      exit(-1);
    }

  if( argc == 2 )
    hexflag = 0;
  else
    hexflag = 1;

  if( ( fp = fopen( argv[hexflag+1], "rt" ) ) == NULL )
    {
      perror("eeprom");
      exit(-3);
    }

  if( ioperm( 0x378, 1, 1 ) < 0 )
    {
      fprintf(stderr, "eeprom: Couldn't allocate port!\n");
      fprintf(stderr, "  Do you have priviledges for the parallel port?\n");
      exit(-2 );
    }

  /* Read input file */
  while( fgets( line, 80, fp ) != NULL )
    {
      if( addr >= 8192 )
	{
	  fprintf(stderr,"eeprom: Address overrun! Aborting write\n");
	  break;
	}
      if( killws( line ) < 0 )
	{
	  fprintf(stderr,"eeprom: malloc() failed!\n" );
	  ioperm( 0x378, 1, 0 );
	  fclose( fp );
	  exit( -3 );
	}

      if( line[0] == ';' || line[0] == '#' || line[0] == '\0' || 
	  line[0] == '\n')
	continue;

      if( line[0] == '@' ) /* Address */
	{
	  addr = str2num( &line[1], hexflag );
	  continue;
	}
      else /* Data */
	{
	  data = str2num( line, hexflag );
	  //	  printf("Write: 0x%x 0x%x\n", addr, data );
	  wordout( addr, data );
	  addr ++;
	}
    }

  ioperm( 0x378, 1, 0 );
  fclose( fp );
}

/* Convert a string to a number
 * Input: str = string to read
 *        hexflag = true if data is in hex
 * Returns: number
 */
unsigned str2num( char *str, char hexflag )
{
  unsigned data;
  int i;

  if( hexflag )
    sscanf( str, "%x", &data );
  else /* Data is in Binary! */
    {
      data = 0;
      for( i = 0; isdigit( str[i] ) ; i ++ )
	{
	  data <<= 1;

	  if( str[i] != '0' )
	    data |= 0x01;
	}
    }

  return data;    
}

/* Remove whitespace from line 
 * Input/Output: str = string to modify
 * Returns: 0 = sucess
 *         -1 = malloc() failure
 */
int killws( char *str )
{
  char *jnk;
  int i, j;

  jnk = (char *)malloc( sizeof( char ) * ( strlen( str ) + 1 ) );
  if( jnk == NULL )
    return -1;

  for( i = 0, j = 0; str[i] != '\0'; i ++ )
    {
      if( ( str[i] != ' ' ) && ( str[i] != '\t' ) )
	{
	  jnk[j] = str[i];
	  j ++;
	}
    }

  jnk[j] = '\0';

  strcpy( str, jnk );

  free( jnk );
}

/* Write a Record to the Programmer
 * Input: address = address of record to store (13 bits) 
 *        data    = data to store in record (8 bits) 
 * Output: via port 0x378 
 * Returns: none 
 */
void wordout( unsigned short address, unsigned char data )
{
  int i;
  unsigned short addr;
  unsigned char dat;

  addr = address;
  dat = data;

  /* Write out address (MSB first) */
  for( i = 12; i >= 0; i -- )
    bitout( ( addr >> i ) & 0x01 );
  
  /* Write out data (MSB first) */
  for( i = 7; i >= 0; i -- )
    bitout( ( data >> i ) & 0x01 );

  /* Clock the data into the EEPROM */
  outb( 0x01, 0x378 );
  iodelay();
  outb( 0x03, 0x378 );
  iodelay();
}

/* Write a bit out to the programmer 
 * Input: bit = data to shift out (used as a conditional)
 * Output: via port 0x378
 * Returns: none 
 */
void bitout( unsigned char bit )
{
  if( bit )
    {
      outb( 0x07, 0x378 );
      iodelay();

      outb( 0x0f, 0x378 );
      iodelay();

      outb( 0x07, 0x378 );
      iodelay();
    }
  else
    {
      outb( 0x03, 0x378 );
      iodelay();

      outb( 0x0b, 0x378 );
      iodelay();

      outb( 0x03, 0x378 );
      iodelay();
    }
}

void iodelay( void )
{
  long i;
  for( i = 0L; i < IODELAY; i ++ );
}
